<?php
// Set the page title before including the header
$page_title = "Manage Deposits";
require_once 'header.php'; // Includes session check, DB connection, and header HTML

// --- ACTION HANDLER ---
// This block processes the 'Approve' or 'Reject' actions from the URL.
if (isset($_GET['action']) && isset($_GET['id'])) {
    $action = $_GET['action'];
    $transaction_id = (int)$_GET['id'];

    // --- APPROVE ACTION ---
    if ($action === 'approve') {
        // Use a database transaction to ensure all steps succeed or none do.
        $conn->begin_transaction();
        try {
            // 1. Get the transaction details, ensuring it's still 'pending'.
            // FOR UPDATE locks the row to prevent two admins from processing it at the same time.
            $sql = "SELECT user_id, amount FROM transactions WHERE id = ? AND status = 'pending' FOR UPDATE";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $transaction_id);
            $stmt->execute();
            $transaction = $stmt->get_result()->fetch_assoc();

            if ($transaction) {
                $user_id = $transaction['user_id'];
                $amount = $transaction['amount'];
                
                // 2. Update the user's wallet balance.
                $update_balance_sql = "UPDATE users SET balance = balance + ? WHERE id = ?";
                $stmt = $conn->prepare($update_balance_sql);
                $stmt->bind_param("di", $amount, $user_id);
                $stmt->execute();

                // 3. Mark this specific transaction as 'completed'.
                $update_txn_sql = "UPDATE transactions SET status = 'completed' WHERE id = ?";
                $stmt = $conn->prepare($update_txn_sql);
                $stmt->bind_param("i", $transaction_id);
                $stmt->execute();

                // --- REFERRAL TRIGGER LOGIC ---
                // After approving, check if this deposit makes the user's referrer eligible for a reward.
                
                // Calculate the new total amount this user has ever deposited successfully.
                $total_deposits_result = $conn->query("SELECT SUM(amount) as total FROM transactions WHERE user_id = {$user_id} AND type = 'deposit' AND status = 'completed'");
                $total_deposits = $total_deposits_result->fetch_assoc()['total'] ?? 0;
                
                // The threshold for the reward is 20 Tk.
                if ($total_deposits >= 20) {
                    // If the threshold is met, find their referral record (if any) and update its status.
                    // This query will only affect the record if it's currently 'pending'.
                    $update_ref_sql = "UPDATE referrals SET status = 'eligible' WHERE referred_id = ? AND status = 'pending'";
                    $stmt_ref = $conn->prepare($update_ref_sql);
                    $stmt_ref->bind_param("i", $user_id);
                    $stmt_ref->execute();
                }
                
                $conn->commit(); // All steps were successful, save the changes.
            }
        } catch (Exception $e) {
            $conn->rollback(); // If any step failed, undo all changes.
            // Optional: Log the error for debugging -> error_log($e->getMessage());
        }
    
    // --- REJECT ACTION ---
    } elseif ($action === 'reject') {
        // Just mark the transaction as 'failed'. No balance or referral changes needed.
        $sql = "UPDATE transactions SET status = 'failed' WHERE id = ? AND status = 'pending'";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $transaction_id);
        $stmt->execute();
    }
    
    // Redirect back to this page to show the updated list and prevent resubmission.
    echo '<script>window.location.href = "deposits.php";</script>';
}

// --- DATA FETCHING ---
// Fetch all manual deposit transactions to display in the table.
$sql = "SELECT t.id, t.amount, t.gateway, t.gateway_trx_id, t.transaction_date, t.status, u.username
        FROM transactions t
        JOIN users u ON t.user_id = u.id
        WHERE t.type = 'deposit' AND t.status = 'pending'
        ORDER BY t.transaction_date DESC";
$transactions_result = $conn->query($sql);
$conn->close();
?>

<style>
    .data-table-container {
        width: 100%;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch; /* Smooth scrolling on iOS */
        margin-bottom: 20px;
        background: var(--card-bg);
        border-radius: 12px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.05);
    }
    
    .data-table {
        width: 100%;
        min-width: 600px; /* Minimum width before scrolling kicks in */
        border-collapse: collapse;
    }
    
    .data-table th, .data-table td {
        padding: 12px 8px;
        text-align: left;
        border-bottom: 1px solid #e5e7eb;
        vertical-align: middle;
    }
    
    .data-table th {
        background: #f9fafb;
        font-size: 12px;
        color: var(--secondary);
        text-transform: uppercase;
        font-weight: 600;
        white-space: nowrap;
    }
    
    .data-table td {
        font-size: 14px;
    }
    
    .status-badge {
        padding: 4px 8px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        color: white;
        text-align: center;
        display: inline-block;
        white-space: nowrap;
    }
    
    .status-completed { background-color: #10B981; }
    .status-pending { background-color: #F59E0B; }
    .status-failed { background-color: #EF4444; }
    
    .action-links {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
    }
    
    .action-btn {
        text-decoration: none;
        padding: 6px 10px;
        border-radius: 6px;
        font-weight: 500;
        font-size: 13px;
        color: white;
        white-space: nowrap;
    }
    
    .btn-approve { background-color: #22C55E; }
    .btn-reject { background-color: #EF4444; }
    
    .no-data {
        text-align: center;
        padding: 30px;
    }
    
    /* Responsive adjustments */
    @media (max-width: 768px) {
        .data-table th, .data-table td {
            padding: 10px 6px;
        }
        
        .action-btn {
            padding: 5px 8px;
            font-size: 12px;
        }
        
        .status-badge {
            padding: 3px 6px;
            font-size: 11px;
        }
    }
    
    @media (max-width: 480px) {
        .data-table th, .data-table td {
            padding: 8px 4px;
            font-size: 13px;
        }
        
        .action-links {
            gap: 4px;
        }
    }
</style>

<div class="data-table-container">
    <table class="data-table">
        <thead>
            <tr>
                <th>User</th>
                <th>Amount</th>
                <th>Gateway</th>
                <th>Trx ID</th>
                <th>Date</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if ($transactions_result && $transactions_result->num_rows > 0): ?>
                <?php while($txn = $transactions_result->fetch_assoc()): ?>
                    <tr>
                        <td><strong><?php echo htmlspecialchars($txn['username']); ?></strong></td>
                        <td>Tk <?php echo htmlspecialchars(number_format($txn['amount'], 2)); ?></td>
                        <td><?php echo htmlspecialchars($txn['gateway']); ?></td>
                        <td><?php echo htmlspecialchars(substr($txn['gateway_trx_id'], 0, 6) . '...'); ?></td>
                        <td><?php echo date("d M, H:i", strtotime($txn['transaction_date'])); ?></td>
                        <td>
                            <span class="status-badge status-<?php echo strtolower(htmlspecialchars($txn['status'])); ?>">
                                <?php echo htmlspecialchars(ucfirst($txn['status'])); ?>
                            </span>
                        </td>
                        <td>
                            <?php if ($txn['status'] === 'pending'): ?>
                                <div class="action-links">
                                    <a href="deposits.php?action=approve&id=<?php echo $txn['id']; ?>" class="action-btn btn-approve" onclick="return confirm('Approve this deposit? This will add funds to the user\'s account.')">Approve</a>
                                    <a href="deposits.php?action=reject&id=<?php echo $txn['id']; ?>" class="action-btn btn-reject" onclick="return confirm('Are you sure you want to reject this deposit?')">Reject</a>
                                </div>
                            <?php else: ?>
                                Processed
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endwhile; ?>
            <?php else: ?>
                <tr>
                    <td colspan="7" class="no-data">No deposit requests found.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php require_once 'footer.php'; ?>