<?php
/*
 * Admin Page for Managing Match Rules
 * /admin/rules.php
 */
$page_title = 'Manage Match Rules';
require_once 'header.php'; // Includes session_start() and db_connect.php

// --- Handle Form Submissions (Create/Update/Delete) ---

// Handle CREATE and UPDATE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_rule'])) {
    $rule_id = $_POST['rule_id'] ?? null;
    $rule_name = trim($_POST['rule_name']);
    $rule_content = trim($_POST['rule_content']);

    if (!empty($rule_name) && !empty($rule_content)) {
        if ($rule_id) { // Update existing rule
            $stmt = $conn->prepare("UPDATE match_rules SET rule_name = ?, rule_content = ? WHERE id = ?");
            $stmt->bind_param("ssi", $rule_name, $rule_content, $rule_id);
            $stmt->execute();
        } else { // Insert new rule
            $stmt = $conn->prepare("INSERT INTO match_rules (rule_name, rule_content) VALUES (?, ?)");
            $stmt->bind_param("ss", $rule_name, $rule_content);
            $stmt->execute();
        }
        // Redirect to avoid form resubmission
        echo '<script>window.location.href = "rules.php";</script>';
    } else {
        $error_message = "Rule name and content cannot be empty.";
    }
}

// Handle DELETE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_rule'])) {
    $rule_id = $_POST['rule_id'];
    $stmt = $conn->prepare("DELETE FROM match_rules WHERE id = ?");
    $stmt->bind_param("i", $rule_id);
    $stmt->execute();
    echo '<script>window.location.href = "rules.php";</script>';
}

// --- Fetch Data for Display ---
$rules = $conn->query("SELECT id, rule_name, rule_content, created_at FROM match_rules ORDER BY id ASC");

// Check if we are editing a specific rule
$edit_rule = null;
if (isset($_GET['edit_id'])) {
    $edit_id = $_GET['edit_id'];
    $stmt = $conn->prepare("SELECT id, rule_name, rule_content FROM match_rules WHERE id = ?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $edit_rule = $result->fetch_assoc();
}
?>
<style>
    .card { background: var(--card-bg); border-radius: 12px; box-shadow: 0 4px 6px rgba(0,0,0,0.05); padding: 25px; margin-bottom: 25px; }
    .form-group { margin-bottom: 20px; }
    .form-group label { display: block; font-weight: 600; margin-bottom: 8px; color: #374151; }
    .form-group input, .form-group textarea {
        width: 100%; padding: 12px; border: 1px solid #D1D5DB; border-radius: 8px;
        font-family: 'Inter', sans-serif; font-size: 16px;
    }
    .form-group textarea { min-height: 150px; resize: vertical; }
    .btn {
        padding: 12px 20px; border: none; border-radius: 8px; font-weight: 600;
        cursor: pointer; transition: background-color 0.2s;
    }
    .btn-primary { background-color: var(--primary); color: white; }
    .btn-primary:hover { background-color: var(--primary-hover); }
    .btn-danger { background-color: #EF4444; color: white; }
    .btn-danger:hover { background-color: #DC2626; }
    .btn-secondary { background-color: #6B7280; color: white; text-decoration: none; }
    .btn-secondary:hover { background-color: #4B5563; }
    .rules-list table { width: 100%; border-collapse: collapse; }
    .rules-list th, .rules-list td { padding: 12px 15px; text-align: left; border-bottom: 1px solid #E5E7EB; }
    .rules-list th { background-color: #F9FAFB; font-weight: 600; color: #374151; }
    .rules-list .actions { display: flex; gap: 10px; }
</style>

<!-- Add/Edit Rule Form Card -->
<div class="card">
    <h2><?php echo $edit_rule ? 'Edit Rule' : 'Add New Rule'; ?></h2>
    <?php if (isset($error_message)) echo "<p style='color: red;'>$error_message</p>"; ?>
    <form action="rules.php" method="post">
        <input type="hidden" name="rule_id" value="<?php echo $edit_rule['id'] ?? ''; ?>">
        <div class="form-group">
            <label for="rule_name">Rule Name (e.g., BR Rules, CS Rules)</label>
            <input type="text" id="rule_name" name="rule_name" value="<?php echo htmlspecialchars($edit_rule['rule_name'] ?? ''); ?>" required>
        </div>
        <div class="form-group">
            <label for="rule_content">Rule Content</label>
            <textarea id="rule_content" name="rule_content" required><?php echo htmlspecialchars($edit_rule['rule_content'] ?? ''); ?></textarea>
        </div>
        <button type="submit" name="save_rule" class="btn btn-primary">
            <?php echo $edit_rule ? 'Update Rule' : 'Save Rule'; ?>
        </button>
        <?php if ($edit_rule): ?>
            <a href="rules.php" class="btn btn-secondary">Cancel Edit</a>
        <?php endif; ?>
    </form>
</div>

<!-- Existing Rules List Card -->
<div class="card rules-list">
    <h2>Existing Match Rules</h2>
    <table>
        <thead>
            <tr>
                <th>Rule Name</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if ($rules->num_rows > 0): ?>
                <?php while($row = $rules->fetch_assoc()): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($row['rule_name']); ?></td>
                        <td class="actions">
                            <a href="rules.php?edit_id=<?php echo $row['id']; ?>" class="btn btn-secondary">Edit</a>
                            <form action="rules.php" method="post" onsubmit="return confirm('Are you sure you want to delete this rule?');">
                                <input type="hidden" name="rule_id" value="<?php echo $row['id']; ?>">
                                <button type="submit" name="delete_rule" class="btn btn-danger">Delete</button>
                            </form>
                        </td>
                    </tr>
                <?php endwhile; ?>
            <?php else: ?>
                <tr>
                    <td colspan="2">No rules have been added yet.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php
$conn->close();
require_once 'footer.php';
?>