<?php
session_start();
require_once 'db_connect.php';


if (!isset($_SESSION['user_id'])) { header("Location: login.php"); exit(); }
$match_id = filter_input(INPUT_GET, 'match_id', FILTER_VALIDATE_INT);
if (!$match_id) { header("Location: home.php"); exit(); }
$user_id = $_SESSION['user_id'];
$error_message = '';


$sql = "SELECT 
            t.id, t.title, t.match_time, t.entry_fee, t.prize_pool, t.max_players, t.category, t.status, t.match_type,
            (SELECT COALESCE(SUM(slots_taken), 0) FROM registrations WHERE tournament_id = t.id) as players_joined,
            (SELECT COUNT(*) FROM registrations WHERE tournament_id = t.id AND user_id = {$user_id}) as is_joined,
            u.balance, u.in_game_name
        FROM tournaments t, users u
        WHERE t.id = ? AND u.id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $match_id, $user_id);
$stmt->execute();
$data = $stmt->get_result()->fetch_assoc();


if (!$data) { die("Error: Match not found."); }
if ($data['status'] !== 'upcoming') { $error_message = 'This match is no longer available for joining.'; }
elseif ($data['is_joined']) { $error_message = 'You have already joined this match.'; }


if ($_SERVER["REQUEST_METHOD"] == "POST" && empty($error_message)) {
    $ign_array = $_POST['ign'] ?? [];
    $ign_array = array_map('trim', $ign_array);
    $ign_array = array_filter($ign_array); 
    $slots_taken = count($ign_array);
    $total_fee = $data['entry_fee'] * $slots_taken;
    $ign_json = json_encode($ign_array);

    
    if ($slots_taken == 0) {
        $error_message = 'Please enter at least one in-game name.';
    } elseif (($data['players_joined'] + $slots_taken) > $data['max_players']) {
        $error_message = 'Sorry, there are not enough slots available for your team.';
    } elseif ($data['balance'] < $total_fee) {
        $error_message = "Insufficient balance. You need Tk $total_fee to join.";
    } else {
        $conn->begin_transaction();
        try {
            
            if (!empty($ign_array[0]) && $ign_array[0] !== $data['in_game_name']) {
                $stmt = $conn->prepare("UPDATE users SET in_game_name = ? WHERE id = ?");
                $stmt->bind_param("si", $ign_array[0], $user_id);
                $stmt->execute();
            }
            
            $new_balance = $data['balance'] - $total_fee;
            $stmt = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
            $stmt->bind_param("di", $new_balance, $user_id);
            $stmt->execute();

            
            $stmt = $conn->prepare("INSERT INTO registrations (user_id, tournament_id, slots_taken, player_ign_data) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("iiis", $user_id, $match_id, $slots_taken, $ign_json);
            $stmt->execute();
            
            
            $log_fee = -$total_fee;
            $desc = "Entry fee for tournament #$match_id";
            $stmt = $conn->prepare("INSERT INTO transactions (user_id, amount, type, description) VALUES (?, ?, 'entry_fee', ?)");
            $stmt->bind_param("ids", $user_id, $log_fee, $desc);
            $stmt->execute();
            
            $conn->commit();
            header("Location: match-list.php?category={$data['category']}&message=joined_successfully");
            exit();
        } catch (Exception $e) {
            $conn->rollback();
            $error_message = "An error occurred: " . $e->getMessage();
        }
    }
}
$conn->close();
?>
<html>
<head>
    <meta charset="UTF-8">
    <title>Hunter BD71 - Join Match</title>

<link rel="icon" type="image/png" href="/images/favicon.png">
    <link rel="apple-touch-icon" href="/images/favicon.png">

    
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
         
        :root {
            --primary-color: #6c5ce7;
            --background-color: #f0f2f5;
            --card-background: #ffffff;
            --text-color: #333;
            --subtle-text: #666;
            --border-color: #e0e0e0;
            --success-color: #2ecc71;
            --error-color: #e74c3c;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
            background-color: var(--background-color);
            margin: 0;
            padding: 15px;
        }
        .container {
            max-width: 500px;
            margin: 0 auto;
        }
        .top-bar {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        .top-bar a {
            color: var(--text-color);
            text-decoration: none;
            font-size: 20px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .card {
            background: var(--card-background);
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        .card h2 {
            margin-top: 0;
            margin-bottom: 8px;
            font-size: 22px;
            color: var(--text-color);
        }
        .card .match-time {
            color: var(--subtle-text);
            font-size: 14px;
            margin-bottom: 20px;
        }
        .details {
            display: flex;
            justify-content: space-between;
            background: #f9f9f9;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            border: 1px solid var(--border-color);
        }
        .details div {
            text-align: center;
        }
        .details span {
            display: block;
            font-size: 13px;
            color: var(--subtle-text);
            margin-bottom: 4px;
        }
        .details strong {
            font-size: 16px;
            color: var(--primary-color);
        }
        .details strong.fee {
            color: var(--error-color);
        }
        label {
            display: block;
            margin-top: 15px;
            font-weight: 600;
            font-size: 14px;
            color: var(--text-color);
        }
        input[type="text"] {
            width: 100%;
            padding: 12px;
            margin-top: 8px;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            font-size: 16px;
            box-sizing: border-box;
            transition: border-color 0.2s;
        }
        input[type="text"]:focus {
            outline: none;
            border-color: var(--primary-color);
        }
        button {
            background: var(--primary-color);
            color: white;
            padding: 14px;
            width: 100%;
            margin-top: 25px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.2s;
        }
        button:hover {
            background: #5a4bd9;
        }
        button:disabled {
            background: #b0a9f2;
            cursor: not-allowed;
        }
        .message {
            text-align: center;
            margin-top: 15px;
            padding: 12px;
            border-radius: 8px;
            font-weight: 500;
        }
        .error {
            color: var(--error-color);
            background-color: #fbe9e7;
        }
        
        .entry-type-selector { display: flex; justify-content: center; gap: 10px; margin-bottom: 25px; }
        .entry-type-selector label { position: relative; }
        .entry-type-selector input[type="radio"] { position: absolute; opacity: 0; }
        .entry-type-selector span { display: block; padding: 10px 20px; border: 2px solid #e0e0e0; border-radius: 8px; cursor: pointer; font-weight: 500; transition: all 0.2s ease; }
        .entry-type-selector input[type="radio"]:checked + span { background-color: var(--primary-color); color: white; border-color: var(--primary-color); }
        .ign-inputs-container .input-group { display: none; }
        .ign-inputs-container .input-group.visible { display: block; animation: fadeIn 0.3s; }
        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
    </style>
</head>
<body>
    <div class="container">
        
        
        <div class="card">
            <?php if (!$data): ?>
                <p class="message error">Invalid match selected.</p>
            <?php else: ?>
                <h2><?php echo htmlspecialchars($data['title']); ?></h2>
                <p class="match-time"><?php echo date("l, d F Y - h:i A", strtotime($data['match_time'])); ?></p>
                <div class="details">
                    <div><span>Win Prize</span><strong>Tk <?php echo htmlspecialchars($data['prize_pool']); ?></strong></div>
                    <div><span>Entry Fee (per player)</span><strong class="fee">Tk <?php echo htmlspecialchars($data['entry_fee']); ?></strong></div>
                </div>
                
                <?php if (!empty($error_message)): ?>
                    <p class="message error"><?php echo $error_message; ?></p>
                <?php else: ?>
                    <form method="POST">
                        
                        <?php $base_fee = $data['entry_fee']; ?>

                        <?php if ($data['match_type'] === 'solo'): ?>
                            <div class="ign-inputs-container" id="ign-container">
                                <div class="input-group visible"><label>Your In-Game Name</label><input type="text" name="ign[]" value="<?php echo htmlspecialchars($data['in_game_name'] ?? ''); ?>" required></div>
                            </div>
                        <?php endif; ?>

                        <?php if ($data['match_type'] === 'duo'): ?>
                            <div class="entry-type-selector">
                                <label><input type="radio" name="entry_type" value="1" onclick="updateForm(1, <?php echo $base_fee; ?>)" checked><span>Join as Solo</span></label>
                                <label><input type="radio" name="entry_type" value="2" onclick="updateForm(2, <?php echo $base_fee; ?>)"><span>Join as Duo</span></label>
                            </div>
                            <div class="ign-inputs-container" id="ign-container">
                                <div class="input-group visible"><label>Player 1 IGN</label><input type="text" name="ign[]" value="<?php echo htmlspecialchars($data['in_game_name'] ?? ''); ?>" required></div>
                                <div class="input-group"><label>Player 2 IGN</label><input type="text" name="ign[]"></div>
                            </div>
                        <?php endif; ?>

                        <?php if ($data['match_type'] === 'squad'): ?>
                            <div class="entry-type-selector">
                                <label><input type="radio" name="entry_type" value="1" onclick="updateForm(1, <?php echo $base_fee; ?>)" checked><span>Solo</span></label>
                                <label><input type="radio" name="entry_type" value="2" onclick="updateForm(2, <?php echo $base_fee; ?>)"><span>Duo</span></label>
                                <label><input type="radio" name="entry_type" value="4" onclick="updateForm(4, <?php echo $base_fee; ?>)"><span>Squad</span></label>
                            </div>
                            <div class="ign-inputs-container" id="ign-container">
                                <div class="input-group visible"><label>Player 1 IGN</label><input type="text" name="ign[]" value="<?php echo htmlspecialchars($data['in_game_name'] ?? ''); ?>" required></div>
                                <div class="input-group"><label>Player 2 IGN</label><input type="text" name="ign[]"></div>
                                <div class="input-group"><label>Player 3 IGN</label><input type="text" name="ign[]"></div>
                                <div class="input-group"><label>Player 4 IGN</label><input type="text" name="ign[]"></div>
                            </div>
                        <?php endif; ?>
                        
                        <button type="submit" id="join-button">Pay Tk <?php echo htmlspecialchars($data['entry_fee']); ?> & Join</button>
                    </form>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
<script>
    const joinButton = document.getElementById('join-button');
    const ignContainer = document.getElementById('ign-container');
    const inputGroups = ignContainer ? ignContainer.querySelectorAll('.input-group') : [];

    function updateForm(playerCount, baseFee) {
        
        const totalFee = playerCount * baseFee;
        
        
        joinButton.textContent = `Pay Tk ${totalFee} & Join`;

        
        inputGroups.forEach((group, index) => {
            const input = group.querySelector('input');
            if (index < playerCount) {
                group.classList.add('visible');
                input.required = true;
            } else {
                group.classList.remove('visible');
                input.required = false;
                input.value = ''; 
            }
        });
    }
</script>
</body>
</html>