<?php
require_once 'header.php'; // Includes session check, db connection, and initial HTML

// --- Get and Validate the Category from the URL ---
$category = isset($_GET['category']) ? trim($_GET['category']) : '';
if (empty($category)) {
    header("Location: tournaments.php");
    exit();
}

// Set the page title dynamically for the header
$page_title = "Matches for: " . strtoupper(htmlspecialchars($category));
echo '<script>document.addEventListener("DOMContentLoaded", function() { document.querySelector(".top-header h1").textContent = "' . addslashes($page_title) . '"; });</script>';

// --- Fetch tournaments for ONLY the selected category ---
$sql = "SELECT 
            t.id, t.title, t.match_time, t.status,
            (SELECT COALESCE(SUM(slots_taken), 0) FROM registrations WHERE tournament_id = t.id) as registered_players
        FROM tournaments t
        WHERE t.category = ?
        ORDER BY t.match_time DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param("s", $category);
$stmt->execute();
$result = $stmt->get_result();

$tournaments = [];
if ($result && $result->num_rows > 0) {
    while($row = $result->fetch_assoc()) {
        $tournaments[] = $row;
    }
}
$conn->close();
?>

<style>
    /* Base styles */
    .page-controls {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        flex-wrap: wrap;
        gap: 10px;
    }
    
    .btn-create, .btn-back {
        color: white;
        padding: 10px 15px;
        border-radius: 8px;
        text-decoration: none;
        font-weight: 500;
        transition: background-color 0.2s;
        font-size: 14px;
        text-align: center;
        flex: 1 1 auto;
        min-width: 120px;
    }
    
    .btn-create {
        background: var(--primary);
    }
    
    .btn-back {
        background: var(--secondary);
    }
    
    .btn-create:hover {
        background: var(--primary-hover);
    }
    
    .data-table-container {
        width: 100%;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        background: var(--card-bg);
        border-radius: 12px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        margin-bottom: 20px;
    }
    
    .data-table {
        width: 100%;
        min-width: 600px;
        border-collapse: collapse;
    }
    
    .data-table th, .data-table td {
        padding: 12px 15px;
        text-align: left;
        border-bottom: 1px solid #e5e7eb;
        vertical-align: middle;
    }
    
    .data-table th {
        background: #f9fafb;
        font-size: 12px;
        color: var(--secondary);
        text-transform: uppercase;
        font-weight: 600;
    }
    
    .data-table td {
        font-size: 14px;
    }
    
    .data-table tr:last-child td {
        border-bottom: none;
    }
    
    .status-label {
        padding: 4px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        color: white;
        text-transform: capitalize;
        display: inline-block;
        min-width: 80px;
        text-align: center;
    }
    
    .status-upcoming { background: #F59E0B; }
    .status-ongoing { background: #3B82F6; }
    .status-completed { background: #10B981; }
    .status-canceled { background: #6B7280; }
    
    .action-links {
        display: flex;
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .action-links a, .action-links button {
        text-decoration: none;
        padding: 6px 10px;
        border-radius: 6px;
        font-weight: 500;
        font-size: 13px;
        white-space: nowrap;
    }
    
    .view-players-link { color: #059669; background: #D1FAE5; }
    .edit-link { color: #2563EB; background: #DBEAFE; }
    .add-results-link { color: #8B5CF6; background: #EDE9FE; }
    .delete-link { 
        color: #DC2626; 
        background: #FEE2E2; 
        border: none; 
        cursor: pointer; 
        font-family: 'Inter', sans-serif; 
        font-size: 13px;
    }
    
    /* Mobile-specific styles */
    @media (max-width: 768px) {
        .page-controls {
            flex-direction: column;
            align-items: stretch;
        }
        
        .btn-create, .btn-back {
            width: 100%;
            padding: 12px;
        }
        
        .data-table th, .data-table td {
            padding: 10px 12px;
            font-size: 13px;
        }
        
        .data-table th {
            font-size: 11px;
        }
        
        .status-label {
            min-width: 70px;
            font-size: 11px;
        }
        
        .action-links {
            gap: 4px;
        }
        
        .action-links a, .action-links button {
            padding: 5px 8px;
            font-size: 12px;
        }
    }
    
    /* Empty state styling */
    .empty-state {
        text-align: center;
        padding: 30px;
        color: #6B7280;
        font-size: 14px;
    }
    
    @media (max-width: 480px) {
        .empty-state {
            padding: 20px;
            font-size: 13px;
        }
    }
</style>

<div class="page-controls">
    <a href="tournaments.php" class="btn-back">← Back to Categories</a>
    <a href="tournament_form.php?category=<?php echo htmlspecialchars($category); ?>" class="btn-create">+ Create New Match</a>
</div>

<div class="data-table-container">
    <table class="data-table">
        <thead>
            <tr>
                <th>Title</th>
                <th>Match Time</th>
                <th>Players</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($tournaments)): ?>
                <tr>
                    <td colspan="5" class="empty-state">No matches found in this category. Click "Create New Match" to add one.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($tournaments as $t): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($t['title']); ?></td>
                        <td><?php echo date("d M Y, h:i A", strtotime($t['match_time'])); ?></td>
                        <td><?php echo htmlspecialchars($t['registered_players']); ?></td>
                        <td>
                            <span class="status-label status-<?php echo htmlspecialchars($t['status']); ?>">
                                <?php echo htmlspecialchars($t['status']); ?>
                            </span>
                        </td>
                        <td class="action-links">
                            <a href="view_registrations.php?id=<?php echo $t['id']; ?>" class="view-players-link">Players</a>
                            <a href="tournament_form.php?id=<?php echo $t['id']; ?>" class="edit-link">Edit</a>
                            <?php if ($t['status'] === 'completed'): ?>
                                <a href="add_winners.php?id=<?php echo $t['id']; ?>" class="add-results-link">Results</a>
                            <?php endif; ?>
                            <form method="POST" action="tournament_delete.php" onsubmit="return confirm('Are you sure you want to permanently delete this match?');">
                                <input type="hidden" name="id" value="<?php echo $t['id']; ?>">
                                <button type="submit" class="delete-link">Delete</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php
require_once 'footer.php';
?>